/*
 *  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
 *
 *  This file is part of the Arnold emulator source code distribution.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "ramrom.h"
#include <memory.h>
#include "emudevice.h"

/*************************/
/**** INICRON RAM-ROM ****/

typedef struct
{
	/* buffer to hold ram contents; split into banks of 16k */
	unsigned char	*RAM;
	/* an array of bits, a bit will be 1 if the corresponding bank is enabled,
	and 0 if the corresponding bank is disabled */
	unsigned char	*BankEnables;
	/* rom select index for eprom onboard ram-rom (not emulated) */
	unsigned char	EPROM_Bank;
	/* flags about the ram-rom status */
	unsigned char	Flags;
	/* number of banks the ram-rom holds */
	unsigned long	NumBlocks;
	/* the mask will be 0 if ram is write disabled and 0x0ffffffff if the ram is write enabled */
	BOOL				WriteEnabled;
	BOOL             Active;
	unsigned long RomSelected;
} RAM_ROM;

static RAM_ROM ramrom;
/* TODO: Need common expansion rom support allowing you to write into the segments

TODO: Eprom support.
TODO: Eprom Segment support for 8 to 64k EPROM
TODO: RAM-ROM enables for 7-15. DONE.
*/
BOOL InicronRAMROM_IsRamOn(void)
{
	return ((ramrom.Flags & RAM_ROM_FLAGS_RAM_ON)!=0);
}

/* set if ram-rom ram is enabled and therefore if roms can be seen */
void InicronRAMROM_SetRamOnState(BOOL State)
{
	if (State)
	{
		ramrom.Flags |= RAM_ROM_FLAGS_RAM_ON;
	}
	else
	{
		ramrom.Flags &= ~RAM_ROM_FLAGS_RAM_ON;
	}

}

/* set read/write state for whole ram */
void InicronRAMROM_SetRamWriteEnableState(BOOL State)
{
	if (State)
	{
		ramrom.Flags |= RAM_ROM_FLAGS_RAM_WRITE_ENABLE;
	}
	else
	{
		ramrom.Flags &= ~RAM_ROM_FLAGS_RAM_WRITE_ENABLE;
	}
}

/* set if eprom on ram-rom is visible */
void InicronRAMROM_SetEPROMOnState(BOOL State)
{
	if (State)
	{
		ramrom.Flags |= RAM_ROM_FLAGS_EPROM_ON;
	}
	else
	{
		ramrom.Flags &= ~RAM_ROM_FLAGS_EPROM_ON;
	}

}

/* true if ram is write enabled, false if ram is write disabled */
BOOL InicronRAMROM_IsRamWriteEnabled(void)
{
	return ((ramrom.Flags & RAM_ROM_FLAGS_RAM_WRITE_ENABLE)!=0);
}

/* true if rom is on and visible, false if off */
BOOL InicronRAMROM_IsEPROMOn(void)
{
	return ((ramrom.Flags & RAM_ROM_FLAGS_EPROM_ON)!=0);
}

/* get selection value for rom */
int		InicronRAMROM_GetEPROMBank(void)
{
	return ramrom.EPROM_Bank;
}

/* initialise, allocate memory and setup */
void InicronRAMROM_Initialise(int NumBlocks)
{
	int Size;

	ramrom.NumBlocks = NumBlocks;

	Size = NumBlocks*16*1024;

	ramrom.RAM = (unsigned char *)malloc(Size);

	if (ramrom.RAM!=NULL)
	{
		memset(ramrom.RAM, 0x0ff, Size);
	}

	Size = (NumBlocks+7)>>3;


	ramrom.BankEnables = (unsigned char *)malloc(Size);

	if (ramrom.BankEnables!=NULL)
	{
		memset(ramrom.BankEnables, 0x0ff, Size);
	}

	ramrom.WriteEnabled = FALSE;
	ramrom.Flags = 0;

}

void InicronRAMROM_Finish(void)
{
	if (ramrom.RAM!=NULL)
	{
		free(ramrom.RAM);
		ramrom.RAM = NULL;
	}

	if (ramrom.BankEnables!=NULL)
	{
		free(ramrom.BankEnables);
		ramrom.BankEnables = NULL;
	}
}

/* set bank enabled  state - if bank is enabled it is visible */
void InicronRAMROM_SetBankEnable(int Bank, BOOL State)
{
	unsigned long BankByte;
	unsigned char BankBit;

	if (ramrom.BankEnables==NULL)
		return;

	BankByte = Bank>>3;
	BankBit = Bank & 0x07;

	if (State)
	{
		ramrom.BankEnables[BankByte] |= (1<<BankBit);
	}
	else
	{
		ramrom.BankEnables[BankByte] &= ~(1<<BankBit);
	}
}

BOOL InicronRAMROM_IsBank0Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(0);
}
BOOL InicronRAMROM_IsBank1Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(1);
}
BOOL InicronRAMROM_IsBank2Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(2);
}
BOOL InicronRAMROM_IsBank3Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(3);
}
BOOL InicronRAMROM_IsBank4Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(4);
}
BOOL InicronRAMROM_IsBank5Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(5);
}
BOOL InicronRAMROM_IsBank6Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(6);
}
BOOL InicronRAMROM_IsBank7Enabled(void)
{
    return InicronRAMROM_GetBankEnableState(7);
}

void InicronRAMROM_SetBank0Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(0, bState);
    InicronRAMROM_SetBankEnable(8, bState);
}

void InicronRAMROM_SetBank1Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(1, bState);
    InicronRAMROM_SetBankEnable(9, bState);
}
void InicronRAMROM_SetBank2Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(2, bState);
    InicronRAMROM_SetBankEnable(10, bState);
}
void InicronRAMROM_SetBank3Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(3, bState);
    InicronRAMROM_SetBankEnable(11, bState);
}
void InicronRAMROM_SetBank4Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(4, bState);
    InicronRAMROM_SetBankEnable(12, bState);
}
void InicronRAMROM_SetBank5Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(5, bState);
    InicronRAMROM_SetBankEnable(13, bState);
}
void InicronRAMROM_SetBank6Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(6, bState);
    InicronRAMROM_SetBankEnable(14, bState);
}
void InicronRAMROM_SetBank7Enable(BOOL bState)
{
    InicronRAMROM_SetBankEnable(7, bState);
    InicronRAMROM_SetBankEnable(15, bState);
}


/* true if bank is enabled, false otherwise */
BOOL InicronRAMROM_GetBankEnableState(int Bank)
{
	unsigned long BankByte;
	unsigned char BankBit;

	if (ramrom.BankEnables==NULL)
		return FALSE;

	BankByte = Bank>>3;
	BankBit = Bank & 0x07;

	return ((ramrom.BankEnables[BankByte] & (1<<BankBit))!=0);
}

void InicronRAMROM_RethinkMemory(MemoryData *pData)
{
    if (ramrom.Active)
    {
        unsigned char *pRamPtr = ramrom.RAM+((ramrom.RomSelected-1)<<14)-0x0c000;
        pData->bRomDisable[7] = TRUE;
        pData->bRomDisable[6] = TRUE;
        pData->pReadPtr[7] = pRamPtr;
        pData->pReadPtr[6] = pRamPtr;
		pData->pReadMaskPtr[6] = GetDefaultReadMask() - 0x0c000;
		pData->pReadMaskPtr[7] = GetDefaultReadMask() - 0x0c000;
		if (ramrom.WriteEnabled)
        {
            pData->pWritePtr[7] = pRamPtr;
            pData->pWritePtr[6] = pRamPtr;
            pData->bRamDisable[7] = TRUE;
            pData->bRamDisable[6] = TRUE;
        }
    }
}

void InicronRAMROM_Install(void)
{
	/* buffer to hold ram contents; split into banks of 16k */
	ramrom.RAM = NULL;
	/* an array of bits, a bit will be 1 if the corresponding bank is enabled,
	and 0 if the corresponding bank is disabled */
	ramrom.BankEnables = NULL;
	/* rom select index for eprom onboard ram-rom (not emulated) */
	ramrom.EPROM_Bank = 0;
	/* flags about the ram-rom status */
	ramrom.Flags = 0;
	
	ramrom.Active = FALSE;
	
	InicronRAMROM_Initialise(16);
}

void	RamRom_Write(Z80_WORD Port, Z80_BYTE Data)
{

    /* only handles ram currently not the rom */

    int BankSelected= Data&0x0f;
    int BankByte;
    int BankBit;

    ramrom.Active = FALSE;

   	ramrom.WriteEnabled = FALSE;

	if (ramrom.RAM==NULL)
		return;

	if ((ramrom.Flags & RAM_ROM_FLAGS_RAM_ON)==0)
		return;

	if (ramrom.Flags & RAM_ROM_FLAGS_RAM_WRITE_ENABLE)
	{
		ramrom.WriteEnabled = TRUE;
	}

    BankByte = BankSelected>>3;
    BankBit = BankSelected & 0x07;

    /* if enabled... */
    if (ramrom.BankEnables[BankByte] & (1<<BankBit))
    {
        ramrom.Active = TRUE;
        ramrom.RomSelected = BankSelected;
    }

    Computer_RethinkMemory();
}


static EmuDeviceSwitch RamRomSwitches[11]=
{
  {
      "Write Enable",                   /* write enable to sram */
	  "WriteEnable",
     InicronRAMROM_IsRamWriteEnabled,
      InicronRAMROM_SetRamWriteEnableState
  },
  {
      "Ram active (socket 0-15)",                     /* if it's active */
	  "RamActive",
     InicronRAMROM_IsRamOn,
      InicronRAMROM_SetRamOnState
  },
  {
      "EPROM is enabled ",              /* if eprom is enabled */
	  "EpromEnabled",
     InicronRAMROM_IsEPROMOn,
      InicronRAMROM_SetEPROMOnState
  },
   {
      "Socket 0/7 enabled",
	  "Socket_0_7_Enabled",
     InicronRAMROM_IsBank0Enabled,
      InicronRAMROM_SetBank0Enable
  },
   {
      "Socket 1/8 enabled",
	  "Socket_1_8_Enabled",
	 InicronRAMROM_IsBank1Enabled,
      InicronRAMROM_SetBank1Enable
  },
   {
      "Socket 2/9 enabled",
	  "Socket_2_9_Enabled",
	 InicronRAMROM_IsBank2Enabled,
      InicronRAMROM_SetBank2Enable
  },
   {
      "Socket 3/10 enabled",
	  "Socket_3_10_Enabled",
	 InicronRAMROM_IsBank3Enabled,
      InicronRAMROM_SetBank3Enable
  },
   {
      "Socket 4/11 enabled",
	  "Socket_4_11_Enabled",
	 InicronRAMROM_IsBank4Enabled,
      InicronRAMROM_SetBank4Enable
  },
   {
      "Socket 5/12 enabled",
	  "Socket_5_12_Enabled",
	 InicronRAMROM_IsBank5Enabled,
      InicronRAMROM_SetBank5Enable
  },
   {
      "Socket 6/13 enabled",
	  "Socket_6_13_Enabled",
	 InicronRAMROM_IsBank6Enabled,
      InicronRAMROM_SetBank6Enable
  },
   {
      "Socket 7/14 enabled",
	  "Socket_7_14_Enabled",
	 InicronRAMROM_IsBank7Enabled,
      InicronRAMROM_SetBank7Enable
  },
};

/* TODO: Verify */
CPCPortWrite RamRomPortWrite[1]=
{
	{
		0x02000,            /* and */
		0x00000,            /* compare */
		RamRom_Write
	}
};

static EmuDevice RamRomDevice=
{
	NULL,
	InicronRAMROM_Install,
	InicronRAMROM_Finish,
	"RAMROM",
	"InicronRamRom",
	"Inicron RAM-ROM",
    CONNECTION_EXPANSION,   /* connects to expansion */
	DEVICE_FLAGS_HAS_EXPANSION_ROMS| DEVICE_FLAGS_FROM_SPECIFICATION, /* has expansion roms */
    0,                /* no read ports */
  NULL,
  1,                    /* 1 write ports */
	RamRomPortWrite,
	0,                /* no memory read*/
	NULL,
	0,                /* no memory write */
	NULL,
	NULL, /* todo: reset?? no reset function */
	InicronRAMROM_RethinkMemory,
  NULL, /* todo: no power function */
	sizeof(RamRomSwitches)/sizeof(RamRomSwitches[0]),                      /* no switches */
	RamRomSwitches,
    0,                      /* no buttons */
    NULL,
	0,
	NULL,
    0,                      /* no onboard roms */
    NULL,
	NULL,	/* no cursor update function */
	NULL,
	NULL,	/* printer */
	NULL,	/* joystick */
	0,
	NULL,	/* memory ranges */
	NULL, /* sound */
	NULL, /* lpen */
	NULL, /* reti */
	NULL, /* ack maskable interrupt */
	NULL, /* dkram data */
	NULL, /* device ram */
	NULL, /* device backup */
	NULL
};

void RamRom_Init(void)
{
 	RegisterDevice(&RamRomDevice);
}
